const os = require('os');
const analytics = require('./analytics');
const logger = require('./logger');

function registerIpcHandlers({
  ipcMain,
  dialog,
  fs,
  fsPromises,
  path,
  crypto,
  app,
  videoUtils,
  settingsStore,
  directoryWatcher,
  ffmpeg,
  getMainWindow,
  shell,
  googleOAuth,
  youtubeService
}) {
  const { videoExtensions, scanDirectory, buildDirectoryStructure } = videoUtils;
  const {
    loadSettings,
    saveSettings,
    getAnalyticsSettings,
    updateAnalyticsPreferences,
    setAnalyticsOptIn,
    rotateAnalyticsUserId
  } = settingsStore;

  function refreshAnalyticsRuntime() {
    if (typeof analytics.applyAnalyticsRuntime !== 'function' || typeof getAnalyticsSettings !== 'function') {
      return;
    }
    try {
      const analyticsSettings = getAnalyticsSettings();
      const config = analytics.getPosthogConfig() || null;
      analytics.applyAnalyticsRuntime({
        analyticsSettings,
        config
      });
    } catch (error) {
      logger.warn('Failed to apply analytics runtime:', error && error.message ? error.message : error);
    }
  }
  ipcMain.handle('analytics.bootstrap', async () => {
    try {
      const analyticsState = getAnalyticsSettings ? getAnalyticsSettings() : { optedIn: false };
      const config = analytics.getPosthogConfig() || null;
      return {
        success: true,
        analytics: analyticsState,
        config
      };
    } catch (error) {
      return {
        success: false,
        error: error && error.message ? error.message : 'Failed to load analytics settings.'
      };
    }
  });

  ipcMain.handle('analytics.preferences.get', async () => {
    try {
      const analyticsState = getAnalyticsSettings ? getAnalyticsSettings() : { optedIn: false };
      return { success: true, analytics: analyticsState };
    } catch (error) {
      return {
        success: false,
        error: error && error.message ? error.message : 'Failed to load analytics preferences.'
      };
    }
  });

  ipcMain.handle('analytics.preferences.set', async (_event, payload = {}) => {
    try {
      if (!payload || typeof payload !== 'object') {
        throw new Error('Invalid payload.');
      }

      if (!Object.prototype.hasOwnProperty.call(payload, 'optedIn')) {
        throw new Error('optedIn flag is required.');
      }

      const rotateUserId = payload.rotateUserId === true;
      const hasRecordingFlag = Object.prototype.hasOwnProperty.call(payload, 'sessionRecordingEnabled');
      const sessionRecordingEnabled = hasRecordingFlag ? Boolean(payload.sessionRecordingEnabled) : undefined;
      const hasRecordingRate = Object.prototype.hasOwnProperty.call(payload, 'sessionRecordingSampleRate');
      const sessionRecordingSampleRate = hasRecordingRate ? payload.sessionRecordingSampleRate : undefined;

      let analyticsState = null;
      if (typeof updateAnalyticsPreferences === 'function') {
        analyticsState = updateAnalyticsPreferences({
          optedIn: true,
          rotateUserId,
          sessionRecordingEnabled,
          sessionRecordingSampleRate
        });
      } else {
        analyticsState = setAnalyticsOptIn ? setAnalyticsOptIn() : { optedIn: true };
        if (rotateUserId && typeof rotateAnalyticsUserId === 'function') {
          analyticsState = rotateAnalyticsUserId();
        }
        if (hasRecordingFlag && analyticsState && analyticsState.sessionRecording) {
          analyticsState.sessionRecording.enabled = sessionRecordingEnabled;
        }
      }

      refreshAnalyticsRuntime();

      return {
        success: true,
        analytics: analyticsState
      };
    } catch (error) {
      return {
        success: false,
        error: error && error.message ? error.message : 'Failed to update analytics preferences.'
      };
    }
  });
  ipcMain.handle('analytics.identity.rotate', async () => {
    try {
      if (typeof updateAnalyticsPreferences !== 'function' && typeof rotateAnalyticsUserId !== 'function') {
        throw new Error('Analytics identity rotation is not supported.');
      }
      const analyticsState = typeof updateAnalyticsPreferences === 'function'
        ? updateAnalyticsPreferences({ rotateUserId: true })
        : rotateAnalyticsUserId();
      refreshAnalyticsRuntime();
      return {
        success: true,
        analytics: analyticsState
      };
    } catch (error) {
      return {
        success: false,
        error: error && error.message ? error.message : 'Failed to rotate analytics identity.'
      };
    }
  });

  ipcMain.handle('analytics.capture', async (_event, payload = {}) => {
    try {
      const result = analytics.queueAnalyticsEvent(payload);
      if (!result.success) {
        throw new Error(result.error || 'Failed to enqueue analytics event.');
      }
      return {
        success: true,
        dropped: Boolean(result.dropped),
        skipped: Boolean(result.skipped),
        reason: result.reason || null,
        diagnosticsId: result.diagnosticsId || null,
        registryVersion: result.registryVersion || null,
        registryUpdatedAt: result.registryUpdatedAt || null
      };
    } catch (error) {
      return {
        success: false,
        error: error && error.message ? error.message : 'Failed to enqueue analytics event.'
      };
    }
  });

  ipcMain.handle('analytics.events.refresh', async () => {
    try {
      const result = await analytics.refreshEventRegistry({ force: true });
      if (!result.success) {
        throw new Error(result.error || 'Failed to refresh analytics registry.');
      }
      return {
        success: true,
        updated: Boolean(result.updated),
        skipped: Boolean(result.skipped)
      };
    } catch (error) {
      return {
        success: false,
        error: error && error.message ? error.message : 'Failed to refresh analytics registry.'
      };
    }
  });

  refreshAnalyticsRuntime();

  const { executeFFmpeg, formatPathForConcat, getThumbnail } = ffmpeg;
  const { startWatchingDirectories, stopAllWatchers } = directoryWatcher;
  const {
    beginAuthentication: beginYouTubeAuthentication,
    getAuthStatus: getYouTubeAuthStatus,
    logout: youtubeLogout,
    listAccounts: listYouTubeAccounts,
    setActiveAccount: setYouTubeActiveAccount,
    removeAccount: removeYouTubeAccount
  } = googleOAuth || {};

  let activeYouTubeAuthSession = null;
  const hasYouTubeService = Boolean(youtubeService && typeof youtubeService.startUpload === 'function');

  ipcMain.handle('youtube.auth.status', async () => {
    if (!getYouTubeAuthStatus) {
      return { success: false, error: 'YouTube authentication is not configured.' };
    }

    try {
      const status = await getYouTubeAuthStatus();
      return { success: true, status };
    } catch (error) {
      return {
        success: false,
        error: error && error.message ? error.message : 'Failed to read YouTube auth status.'
      };
    }
  });

  ipcMain.handle('youtube.auth.start', async (event) => {
    if (!beginYouTubeAuthentication) {
      return { success: false, error: 'YouTube authentication is not configured.' };
    }

    if (activeYouTubeAuthSession) {
      return { success: false, error: 'Authentication is already in progress.' };
    }

    try {
      const session = await beginYouTubeAuthentication();
      activeYouTubeAuthSession = session;
      const authUrl = session.authUrl;
      let browserOpened = false;
      const targetWindow = event.sender;

      const emitToRenderer = (payload) => {
        const primaryTarget = targetWindow && !targetWindow.isDestroyed() ? targetWindow : null;
        if (primaryTarget) {
          primaryTarget.send('youtube.auth.completed', payload);
          return;
        }
        const fallbackWindow = getMainWindow ? getMainWindow() : null;
        if (fallbackWindow && fallbackWindow.webContents && !fallbackWindow.webContents.isDestroyed()) {
          fallbackWindow.webContents.send('youtube.auth.completed', payload);
        }
      };

      session.waitForTokens().then(async () => {
        activeYouTubeAuthSession = null;
        if (getYouTubeAuthStatus) {
          try {
            const status = await getYouTubeAuthStatus();
            emitToRenderer({ success: true, status });
          } catch (statusError) {
            emitToRenderer({
              success: false,
              error: statusError && statusError.message ? statusError.message : 'Failed to finalize YouTube authentication.'
            });
          }
        } else {
          emitToRenderer({ success: true, status: { authenticated: true } });
        }
      }).catch((error) => {
        activeYouTubeAuthSession = null;
        emitToRenderer({
          success: false,
          error: error && error.message ? error.message : 'YouTube authentication failed.'
        });
      });

      if (authUrl) {
        if (shell && typeof shell.openExternal === 'function') {
          try {
            const maybePromise = shell.openExternal(authUrl);
            if (maybePromise && typeof maybePromise.then === 'function') {
              await maybePromise;
            }
            browserOpened = true;
          } catch (error) {
            logger.warn('Failed to open browser automatically for YouTube auth:', error);
          }
        }

        const targetWindow = event.sender;
        session.waitForTokens().then(async () => {
          activeYouTubeAuthSession = null;
          const status = await getYouTubeAuthStatus();
          const recipient = !targetWindow.isDestroyed()
            ? targetWindow
            : (() => {
                const mainWindow = getMainWindow();
                return mainWindow ? mainWindow.webContents : null;
              })();
          recipient?.send('youtube.auth.completed', { success: true, status });
        }).catch((error) => {
          activeYouTubeAuthSession = null;
          const recipient = !targetWindow.isDestroyed()
            ? targetWindow
            : (() => {
                const mainWindow = getMainWindow();
                return mainWindow ? mainWindow.webContents : null;
              })();
          recipient?.send('youtube.auth.completed', {
            success: false,
            error: error && error.message ? error.message : 'YouTube authentication failed.'
          });
        });
      }

      return {
        success: true,
        authUrl,
        browserOpened
      };
    } catch (error) {
      activeYouTubeAuthSession = null;
      return {
        success: false,
        error: error && error.message ? error.message : 'Failed to initiate YouTube authentication.'
      };
    }
  });

  ipcMain.handle('youtube.auth.cancel', async () => {
    if (!activeYouTubeAuthSession) {
      return { success: true, canceled: false };
    }

    try {
      if (typeof activeYouTubeAuthSession.cancel === 'function') {
        await activeYouTubeAuthSession.cancel();
      }
    } finally {
      activeYouTubeAuthSession = null;
    }

    return { success: true, canceled: true };
  });

  ipcMain.handle('youtube.auth.logout', async (_event, payload = {}) => {
    const targetAccountId = payload && typeof payload === 'object' ? payload.accountId || null : null;

    if (youtubeService && typeof youtubeService.removeAccount === 'function') {
      try {
        const status = await youtubeService.removeAccount(targetAccountId);
        return { success: true, status };
      } catch (error) {
        return {
          success: false,
          error: error && error.message ? error.message : 'Failed to disconnect YouTube account.'
        };
      }
    }

    if (!youtubeLogout) {
      return { success: false, error: 'YouTube authentication is not configured.' };
    }

    try {
      await youtubeLogout(targetAccountId);
      const status = getYouTubeAuthStatus ? await getYouTubeAuthStatus() : { authenticated: false };
      return { success: true, status };
    } catch (error) {
      return {
        success: false,
        error: error && error.message ? error.message : 'Failed to disconnect YouTube account.'
      };
    }
  });

  ipcMain.handle('youtube.upload.start', async (event, payload) => {
    if (!hasYouTubeService) {
      return { success: false, error: 'YouTube integration is unavailable.' };
    }

    const { videoPath, metadata, accountId } = payload || {};
    if (!videoPath) {
      return { success: false, error: 'videoPath is required.' };
    }

    try {
      const job = await youtubeService.startUpload({
        videoPath,
        metadata,
        webContents: event.sender,
        accountId
      });
      return { success: true, job };
    } catch (error) {
      return {
        success: false,
        error: error && error.message ? error.message : 'Failed to start YouTube upload.'
      };
    }
  });

  ipcMain.handle('youtube.video.updateMetadata', async (event, payload) => {
    if (!hasYouTubeService) {
      return { success: false, error: 'YouTube integration is unavailable.' };
    }

    const { videoPath, videoId, metadata } = payload || {};
    // Require either videoPath or videoId
    if (!videoPath && !videoId) {
      return { success: false, error: 'Either videoPath or videoId is required.' };
    }

    try {
      const videoRecord = await youtubeService.updateMetadata({
        videoPath,
        videoId,
        metadata,
        webContents: event.sender
      });
      return { success: true, video: videoRecord };
    } catch (error) {
      return {
        success: false,
        error: error && error.message ? error.message : 'Failed to update YouTube metadata.'
      };
    }
  });

  ipcMain.handle('youtube.video.remove', async (event, payload) => {
    if (!hasYouTubeService) {
      return { success: false, error: 'YouTube integration is unavailable.' };
    }

    const { videoPath, videoId, accountId } = payload || {};
    // Require either videoPath or videoId for YouTube library videos
    if (!videoPath && !videoId) {
      return { success: false, error: 'Either videoPath or videoId is required.' };
    }

    try {
      await youtubeService.deleteVideo({
        videoPath,
        videoId,
        accountId,
        webContents: event.sender
      });
      return { success: true };
    } catch (error) {
      return {
        success: false,
        error: error && error.message ? error.message : 'Failed to remove YouTube video.'
      };
    }
  });

  ipcMain.handle('youtube.video.list', async (_event, payload = {}) => {
    if (!hasYouTubeService) {
      return { success: false, error: 'YouTube integration is unavailable.' };
    }

    try {
      // Get accountId from payload first (from renderer), then fall back to active account
      let accountId = payload.accountId || null;
      
      if (!accountId && googleOAuth && typeof googleOAuth.getAuthStatus === 'function') {
        try {
          const status = await googleOAuth.getAuthStatus();
          accountId = status && status.activeAccountId ? status.activeAccountId : null;
        } catch (error) {
          // Ignore errors
        }
      }
      
      if (!accountId) {
        // No account ID - return empty records
        return { success: true, videos: {} };
      }
      
      // Check if store is already loaded - if so, return immediately without calling getAllRecords
      const store = youtubeService.getStore(accountId);
      if (store && store.loaded) {
        const videos = { ...store.videos };
        return { success: true, videos };
      }
      
      // Store not loaded - load it now (this loads from disk, not YouTube API)
      const videos = await youtubeService.getAllRecords(accountId);
      return { success: true, videos };
    } catch (error) {
      return {
        success: false,
        error: error && error.message ? error.message : 'Failed to load YouTube records.'
      };
    }
  });

  ipcMain.handle('youtube.library.list', async (_event, payload = {}) => {
    if (!hasYouTubeService) {
      return { success: false, error: 'YouTube integration is unavailable.' };
    }

    const forceValue = payload && payload.force === true;
    logger.log('[IPC] youtube.library.list called with payload.force=', payload && payload.force, 'resolved force=', forceValue, 'accountId=', payload && payload.accountId);

    try {
      const snapshot = await youtubeService.getLibrarySnapshot({
        maxVideos: payload && Number.isFinite(payload.maxVideos) ? payload.maxVideos : undefined,
        maxPlaylists: payload && Number.isFinite(payload.maxPlaylists) ? payload.maxPlaylists : undefined,
        maxPlaylistItems: payload && Number.isFinite(payload.maxPlaylistItems) ? payload.maxPlaylistItems : undefined,
        force: forceValue,
        accountId: payload && payload.accountId ? payload.accountId : undefined
      });
      return {
        success: true,
        accountId: snapshot.accountId || null,
        channelId: snapshot.channelId || null,
        channelTitle: snapshot.channelTitle || null,
        fetchedAt: snapshot.fetchedAt || null,
        videos: snapshot.videos || [],
        playlists: snapshot.playlists || [],
        playlistItems: snapshot.playlistItems || {}
      };
    } catch (error) {
      return {
        success: false,
        error: error && error.message ? error.message : 'Failed to load YouTube library.'
      };
    }
  });

  ipcMain.handle('youtube.cache.load', async (_event, payload = {}) => {
    if (!hasYouTubeService) {
      return { success: false, error: 'YouTube integration is unavailable.' };
    }

    const accountId = payload && payload.accountId ? payload.accountId : null;
    if (!accountId) {
      return { success: false, error: 'accountId is required.' };
    }

    logger.log('[IPC] youtube.cache.load called for account:', accountId);
    try {
      const cache = await youtubeService.loadLibraryCache(accountId);
      if (cache) {
        logger.log('[IPC] youtube.cache.load: Cache found with', cache.videos ? cache.videos.length : 0, 'videos');
        return {
          success: true,
          cache: {
            accountId: cache.accountId || accountId,
            channelId: cache.channelId || null,
            channelTitle: cache.channelTitle || null,
            fetchedAt: cache.fetchedAt || null,
            videos: cache.videos || [],
            playlists: cache.playlists || [],
            playlistItems: cache.playlistItems || {}
          }
        };
      }
      logger.log('[IPC] youtube.cache.load: No cache found for account:', accountId);
      return { success: true, cache: null };
    } catch (error) {
      logger.log('[IPC] youtube.cache.load: Error loading cache:', error && error.message ? error.message : error);
      return {
        success: false,
        error: error && error.message ? error.message : 'Failed to load YouTube cache.'
      };
    }
  });

  ipcMain.handle('youtube.cache.addVideo', async (_event, payload = {}) => {
    if (!hasYouTubeService) {
      return { success: false, error: 'YouTube integration is unavailable.' };
    }

    const accountId = payload && payload.accountId ? payload.accountId : null;
    const videoData = payload && payload.video ? payload.video : null;

    if (!accountId || !videoData) {
      return { success: false, error: 'accountId and video are required.' };
    }

    try {
      const success = await youtubeService.addVideoToCache(accountId, videoData);
      return { success };
    } catch (error) {
      return {
        success: false,
        error: error && error.message ? error.message : 'Failed to add video to cache.'
      };
    }
  });

  ipcMain.handle('youtube.accounts.list', async () => {
    try {
      let status;
      if (youtubeService && typeof youtubeService.listAccounts === 'function') {
        status = await youtubeService.listAccounts();
      } else {
        const authStatus = getYouTubeAuthStatus ? await getYouTubeAuthStatus() : null;
        const accounts = listYouTubeAccounts ? await listYouTubeAccounts() : [];
        status = authStatus
          ? {
              ...authStatus,
              accounts: Array.isArray(authStatus.accounts) && authStatus.accounts.length > 0 ? authStatus.accounts : accounts
            }
          : {
              authenticated: accounts.length > 0,
              activeAccountId: accounts.length > 0 && accounts[0] && accounts[0].id ? accounts[0].id : null,
              accounts
            };
      }

      // Preload stores for all authenticated accounts - wait for active account to complete
      if (status && status.authenticated && Array.isArray(status.accounts)) {
        // Preload active account first and wait for it to complete
        if (status.activeAccountId && youtubeService && typeof youtubeService.preloadStore === 'function') {
          try {
            await youtubeService.preloadStore(status.activeAccountId);
          } catch {
            // Ignore preload errors - stores will be loaded when needed
          }
        }
        // Preload other accounts in the background (don't wait)
        for (const account of status.accounts) {
          if (account && account.id && account.id !== status.activeAccountId && youtubeService && typeof youtubeService.preloadStore === 'function') {
            youtubeService.preloadStore(account.id).catch(() => {
              // Ignore preload errors
            });
          }
        }
      }

      return {
        success: true,
        status
      };
    } catch (error) {
      return {
        success: false,
        error: error && error.message ? error.message : 'Failed to list YouTube accounts.'
      };
    }
  });

  ipcMain.handle('youtube.accounts.setActive', async (_event, payload = {}) => {
    const targetAccountId = payload && typeof payload === 'object' ? payload.accountId || null : null;
    if (!targetAccountId) {
      return { success: false, error: 'accountId is required.' };
    }

    try {
      if (youtubeService && typeof youtubeService.selectAccount === 'function') {
        const status = await youtubeService.selectAccount(targetAccountId);
        return { success: true, status };
      }

      if (setYouTubeActiveAccount) {
        await setYouTubeActiveAccount(targetAccountId);
      }
      const status = getYouTubeAuthStatus ? await getYouTubeAuthStatus() : { authenticated: true, activeAccountId: targetAccountId };
      return { success: true, status };
    } catch (error) {
      return {
        success: false,
        error: error && error.message ? error.message : 'Failed to select YouTube account.'
      };
    }
  });

  ipcMain.handle('youtube.accounts.remove', async (_event, payload = {}) => {
    const targetAccountId = payload && typeof payload === 'object' ? payload.accountId || null : null;

    try {
      if (youtubeService && typeof youtubeService.removeAccount === 'function') {
        const status = await youtubeService.removeAccount(targetAccountId);
        return { success: true, status };
      }

      if (!youtubeLogout) {
        return { success: false, error: 'YouTube authentication is not configured.' };
      }

      await youtubeLogout(targetAccountId);
      const status = getYouTubeAuthStatus ? await getYouTubeAuthStatus() : { authenticated: false };
      return { success: true, status };
    } catch (error) {
      return {
        success: false,
        error: error && error.message ? error.message : 'Failed to remove YouTube account.'
      };
    }
  });

  ipcMain.handle('select-video-files', async () => {
    const result = await dialog.showOpenDialog({
      properties: ['openFile', 'multiSelections'],
      filters: [
        { name: 'Video Files', extensions: videoExtensions },
        { name: 'All Files', extensions: ['*'] }
      ]
    });
    return result.filePaths;
  });

  ipcMain.handle('copy-video-file', async (event, { sourcePath, targetDirectory }) => {
    try {
      if (!sourcePath || !targetDirectory) {
        throw new Error('Source and target paths are required');
      }

      if (!fs.existsSync(sourcePath)) {
        throw new Error('Source file does not exist');
      }

      const sourceStat = fs.statSync(sourcePath);
      if (!sourceStat.isFile()) {
        throw new Error('Selected item is not a file');
      }

      if (!fs.existsSync(targetDirectory)) {
        fs.mkdirSync(targetDirectory, { recursive: true });
      }

      const targetStat = fs.statSync(targetDirectory);
      if (!targetStat.isDirectory()) {
        throw new Error('Target path is not a directory');
      }

      const originalName = path.basename(sourcePath);
      let destinationPath = await getUniqueDestinationPath(targetDirectory, originalName);

      if (path.resolve(sourcePath) === path.resolve(destinationPath)) {
        return { success: true, destinationPath };
      }

      await fsPromises.copyFile(sourcePath, destinationPath);

      return { success: true, destinationPath };
    } catch (error) {
      logger.error('Error copying video file:', error);
      return { success: false, error: error.message || 'Unknown error' };
    }
  });

  ipcMain.handle('delete-video-file', async (event, filePath) => {
    try {
      if (!filePath) {
        throw new Error('File path is required');
      }

      if (!fs.existsSync(filePath)) {
        throw new Error('File does not exist');
      }

      const stats = fs.statSync(filePath);
      if (!stats.isFile()) {
        throw new Error('Selected path is not a file');
      }

      await fsPromises.unlink(filePath);

      try {
        const thumbnailsDir = path.join(app.getPath('userData'), 'thumbnails');
        const videoHash = crypto.createHash('md5').update(filePath).digest('hex');
        const thumbnailPath = path.join(thumbnailsDir, `${videoHash}.jpg`);
        if (fs.existsSync(thumbnailPath)) {
          await fsPromises.unlink(thumbnailPath);
        }
      } catch (thumbError) {
        logger.warn('Failed to remove thumbnail for deleted video:', thumbError.message || thumbError);
      }

      return { success: true };
    } catch (error) {
      logger.error('Error deleting video file:', error);
      return { success: false, error: error.message || 'Unknown error' };
    }
  });

  ipcMain.handle('rename-video-file', async (event, { filePath, newFileName }) => {
    try {
      if (!filePath) {
        throw new Error('File path is required');
      }

      if (!newFileName) {
        throw new Error('New file name is required');
      }

      const trimmedName = newFileName.trim();
      if (!trimmedName) {
        throw new Error('New file name cannot be empty');
      }

      if (/[\\/:*?"<>|]/.test(trimmedName)) {
        throw new Error('File name contains invalid characters (\\ / : * ? " < > |)');
      }

      if (!fs.existsSync(filePath)) {
        throw new Error('File does not exist');
      }

      const originalStats = fs.statSync(filePath);
      if (!originalStats.isFile()) {
        throw new Error('Selected path is not a file');
      }

      const directoryOfFile = path.dirname(filePath);
      const targetPath = path.join(directoryOfFile, trimmedName);

      const resolvedSource = path.resolve(filePath);
      const resolvedTarget = path.resolve(targetPath);
      const isSamePath = resolvedSource === resolvedTarget;

      if (!isSamePath && fs.existsSync(targetPath)) {
        throw new Error('A file with that name already exists');
      }

      if (!isSamePath) {
        await fsPromises.rename(filePath, targetPath);

        const thumbnailsDir = path.join(app.getPath('userData'), 'thumbnails');
        const oldHash = crypto.createHash('md5').update(filePath).digest('hex');
        const oldThumbnailPath = path.join(thumbnailsDir, `${oldHash}.jpg`);
        if (fs.existsSync(oldThumbnailPath)) {
          try {
            await fsPromises.unlink(oldThumbnailPath);
          } catch (thumbError) {
            logger.warn('Failed to remove thumbnail for renamed video:', thumbError.message || thumbError);
          }
        }
      }

      const finalPath = isSamePath ? filePath : targetPath;
      const finalStats = fs.statSync(finalPath);
      const extension = path.extname(finalPath).slice(1).toLowerCase();

      const settings = loadSettings();
      let monitoredDirPath = null;
      let monitoredDirName = null;

      const normalizedFinalPath = path.resolve(finalPath);
      const comparisonTarget = process.platform === 'win32' ? normalizedFinalPath.toLowerCase() : normalizedFinalPath;

      if (settings.monitoredDirectories && settings.monitoredDirectories.length > 0) {
        for (const dirObj of settings.monitoredDirectories) {
          const dirPath = typeof dirObj === 'string' ? dirObj : dirObj.path;
          const normalizedDirPath = path.resolve(dirPath);
          const comparisonDir = process.platform === 'win32' ? normalizedDirPath.toLowerCase() : normalizedDirPath;
          const dirPrefix = comparisonDir.endsWith(path.sep) ? comparisonDir : comparisonDir + path.sep;
          if (comparisonTarget === comparisonDir || comparisonTarget.startsWith(dirPrefix)) {
            monitoredDirPath = dirPath;
            monitoredDirName = typeof dirObj === 'string' ? path.basename(dirPath) : dirObj.name;
            break;
          }
        }
      }

      let relativePath = path.basename(finalPath);
      let directoryRelative = '';

      if (monitoredDirPath) {
        relativePath = path.relative(monitoredDirPath, finalPath);
        directoryRelative = path.dirname(relativePath);
        if (directoryRelative === '.' || directoryRelative === path.sep) {
          directoryRelative = '';
        }
      }

      const updatedVideo = {
        path: finalPath,
        name: path.basename(finalPath),
        relativePath,
        directory: directoryRelative,
        size: finalStats.size,
        created: finalStats.birthtime,
        modified: finalStats.mtime,
        extension
      };

      if (monitoredDirPath) {
        updatedVideo.directoryPath = monitoredDirPath;
      }

      if (typeof monitoredDirName === 'string') {
        updatedVideo.directoryName = monitoredDirName;
      }

      return {
        success: true,
        updatedVideo
      };
    } catch (error) {
      logger.error('Error renaming video file:', error);
      return { success: false, error: error.message || 'Unknown error' };
    }
  });

  ipcMain.handle('export-video-splits', async (event, payload) => {
    let tempDir = null;

    try {
      if (!payload || !payload.videoPath || !Array.isArray(payload.segments) || payload.segments.length === 0) {
        return { success: false, error: 'No segments provided for export.' };
      }

      const sourcePath = payload.videoPath;

      if (!fs.existsSync(sourcePath)) {
        return { success: false, error: 'Source video file not found.' };
      }

      const segments = payload.segments
        .map((segment) => ({
          start: Number(segment.start),
          end: Number(segment.end)
        }))
        .filter((segment) => Number.isFinite(segment.start) && Number.isFinite(segment.end) && segment.end > segment.start + 0.01)
        .sort((a, b) => a.start - b.start);

      if (segments.length === 0) {
        return { success: false, error: 'No valid segments to export.' };
      }

      const originalExt = path.extname(sourcePath) || '';
      const normalizedExt = originalExt || '.mp4';
      const extensionWithoutDot = normalizedExt.startsWith('.') ? normalizedExt.slice(1) : normalizedExt;
      const baseName = path.basename(sourcePath, originalExt);
      const defaultFileName = `${baseName || 'edited-video'}-edited${normalizedExt}`;

      const saveResult = await dialog.showSaveDialog({
        title: 'Export Edited Video',
        defaultPath: path.join(path.dirname(sourcePath), defaultFileName),
        filters: [
          { name: 'Video', extensions: [extensionWithoutDot || 'mp4'] },
          { name: 'All Files', extensions: ['*'] }
        ]
      });

      if (saveResult.canceled || !saveResult.filePath) {
        return { success: false, canceled: true };
      }

      tempDir = await fsPromises.mkdtemp(path.join(os.tmpdir(), 'miba-video-edit-'));
      const segmentOutputs = [];

      for (let i = 0; i < segments.length; i += 1) {
        const segment = segments[i];
        const durationSeconds = Number((segment.end - segment.start).toFixed(3));
        const startSeconds = Number(segment.start.toFixed(3));
        const tempOutput = path.join(tempDir, `segment_${i}${normalizedExt}`);

        const commonArgs = [
          '-y',
          '-hide_banner',
          '-loglevel', 'error',
          '-ss', startSeconds.toString(),
          '-i', sourcePath,
          '-t', durationSeconds.toString(),
          '-avoid_negative_ts', 'make_zero'
        ];

        try {
          await executeFFmpeg([...commonArgs, '-c', 'copy', tempOutput]);
        } catch (copyError) {
          await executeFFmpeg([
            ...commonArgs,
            '-c:v', 'libx264',
            '-preset', 'faster',
            '-crf', '18',
            '-c:a', 'aac',
            '-b:a', '192k',
            '-movflags', 'faststart',
            tempOutput
          ]);
        }

        if (!fs.existsSync(tempOutput)) {
          throw new Error('Failed to create temporary segment file.');
        }

        segmentOutputs.push(tempOutput);
      }

      const outputPath = saveResult.filePath;

      if (segmentOutputs.length === 1) {
        await fsPromises.copyFile(segmentOutputs[0], outputPath);
      } else {
        const listFile = path.join(tempDir, 'concat_list.txt');
        const listContent = segmentOutputs
          .map((file) => `file '${formatPathForConcat(file)}'`)
          .join(os.EOL);

        await fsPromises.writeFile(listFile, listContent, 'utf8');

        await executeFFmpeg([
          '-y',
          '-hide_banner',
          '-loglevel', 'error',
          '-f', 'concat',
          '-safe', '0',
          '-i', listFile,
          '-c', 'copy',
          outputPath
        ]);
      }

      return { success: true, outputPath };
    } catch (error) {
      logger.error('Error exporting video segments:', error);
      return { success: false, error: error.message || 'Failed to export video.' };
    } finally {
      if (tempDir) {
        try {
          await fsPromises.rm(tempDir, { recursive: true, force: true });
        } catch (rmError) {
          try {
            await fsPromises.rmdir(tempDir, { recursive: true });
          } catch (cleanupError) {
            logger.warn('Failed to remove temporary directory:', cleanupError.message || cleanupError);
          }
        }
      }
    }
  });

  ipcMain.handle('get-file-stats', async (event, filePath) => {
    try {
      const stats = fs.statSync(filePath);
      return {
        size: stats.size,
        created: stats.birthtime,
        modified: stats.mtime
      };
    } catch (error) {
      return null;
    }
  });

  ipcMain.handle('select-directory', async () => {
    const result = await dialog.showOpenDialog({
      properties: ['openDirectory']
    });
    return result.filePaths[0] || null;
  });

  ipcMain.handle('show-save-dialog', async (_event, options) => {
    const result = await dialog.showSaveDialog(options);
    return result.canceled ? { canceled: true } : { filePath: result.filePath };
  });

  ipcMain.handle('show-open-dialog', async (_event, options) => {
    const result = await dialog.showOpenDialog(options);
    return result.canceled ? { canceled: true } : { filePaths: result.filePaths };
  });

  ipcMain.handle('get-settings', () => {
    return loadSettings();
  });

  ipcMain.handle('save-settings', (event, settings) => {
    saveSettings(settings);
    stopAllWatchers();
    if (settings.monitoredDirectories && settings.monitoredDirectories.length > 0) {
      startWatchingDirectories(settings.monitoredDirectories);
      const mainWindow = getMainWindow();
      if (mainWindow) {
        const directoryStructure = buildDirectoryStructure(settings.monitoredDirectories);
        mainWindow.webContents.send('directory-structure-updated', directoryStructure);
      }
    }
    return true;
  });

  ipcMain.handle('get-directory-structure', () => {
    const settings = loadSettings();
    if (settings.monitoredDirectories && settings.monitoredDirectories.length > 0) {
      return buildDirectoryStructure(settings.monitoredDirectories);
    }
    return [];
  });

  ipcMain.handle('scan-directory', async (event, dirPath) => {
    if (!dirPath || !fs.existsSync(dirPath)) {
      return [];
    }
    return scanDirectory(dirPath);
  });

  ipcMain.handle('get-thumbnail', async (event, videoPath) => {
    try {
      const thumbnailPath = await getThumbnail(videoPath);
      return thumbnailPath;
    } catch (error) {
      return null;
    }
  });

  const { readCuePoints } = require('./cuePoints');
  const { getVideoMetadata } = require('./videoMetadata');
  const { detectHighlights, detectHighlightsWithTemplate, detectCombinedHighlights, getWaveformData } = require('./audioHighlightDetector');
  const { extractTemplateFromRegion, detectVisualHighlights } = require('./visualHighlightDetector');
  const { saveTemplate, loadTemplates, loadTemplateByName, deleteTemplate, exportTemplate, importTemplate } = require('./highlightTemplate');

  ipcMain.handle('read-cue-points', async (event, videoPath) => {
    logger.log('[IPC] read-cue-points called with path:', videoPath);
    try {
      if (!videoPath) {
        logger.warn('[IPC] read-cue-points: No video path provided');
        return { success: false, error: 'Video path is required', cuePoints: [] };
      }

      const cuePoints = await readCuePoints(videoPath);
      logger.log('[IPC] read-cue-points: Returning', cuePoints.length, 'cue points');
      return { success: true, cuePoints };
    } catch (error) {
      logger.error('[IPC] Error reading cue points via IPC:', error);
      logger.error('[IPC] Error stack:', error.stack);
      return {
        success: false,
        error: error && error.message ? error.message : 'Failed to read cue points',
        cuePoints: []
      };
    }
  });

  ipcMain.handle('get-video-metadata', async (_event, videoPath) => {
    try {
      if (!videoPath) {
        return { success: false, error: 'Video path is required', metadata: null };
      }
      const metadata = await getVideoMetadata(videoPath);
      if (!metadata) {
        return { success: false, error: 'Could not read video metadata. The file may be unsupported or inaccessible.', metadata: null };
      }
      return { success: true, metadata };
    } catch (error) {
      logger.error('[IPC] get-video-metadata error:', error);
      return { success: false, error: error.message || 'Failed to get video metadata.', metadata: null };
    }
  });

  ipcMain.handle('get-waveform-data', async (_event, payload) => {
    const { videoPath, duration } = payload || {};
    try {
      if (!videoPath) {
        return { success: false, error: 'videoPath is required.' };
      }
      return await getWaveformData(videoPath, Number(duration) || 0);
    } catch (error) {
      logger.error('[IPC] get-waveform-data error:', error);
      return { success: false, error: error.message || 'Failed to get waveform data.' };
    }
  });

  ipcMain.handle('detect-audio-highlights', async (event, payload) => {
    const { videoPath, templateStart, templateEnd, similarityThreshold, minGapSeconds } = payload || {};
    const webContents = event.sender;

    const sendProgress = (percent, message) => {
      try {
        if (webContents && !webContents.isDestroyed()) {
          webContents.send('audio-scan-progress', { percent, message });
        }
      } catch (err) {
        logger.warn('[IPC] Failed to send audio-scan-progress:', err && err.message);
      }
    };

    try {
      if (!videoPath || templateStart == null || templateEnd == null) {
        return { success: false, error: 'videoPath, templateStart, and templateEnd are required.' };
      }

      const result = await detectHighlights({
        videoPath,
        templateStart: Number(templateStart),
        templateEnd: Number(templateEnd),
        similarityThreshold: similarityThreshold != null ? Number(similarityThreshold) : undefined,
        minGapSeconds: minGapSeconds != null ? Number(minGapSeconds) : undefined,
        onProgress: sendProgress
      });

      return result;
    } catch (error) {
      logger.error('[IPC] detect-audio-highlights error:', error);
      return {
        success: false,
        error: error.message || 'Audio highlight detection failed.'
      };
    }
  });

  ipcMain.handle('detect-audio-highlights-with-template', async (event, payload) => {
    const { videoPath, audioSampleBase64, similarityThreshold, minGapSeconds } = payload || {};
    const webContents = event.sender;

    const sendProgress = (percent, message) => {
      try {
        if (webContents && !webContents.isDestroyed()) {
          webContents.send('audio-scan-progress', { percent, message });
        }
      } catch (err) {
        logger.warn('[IPC] Failed to send audio-scan-progress:', err && err.message);
      }
    };

    try {
      if (!videoPath || !audioSampleBase64) {
        return { success: false, error: 'videoPath and audioSampleBase64 are required.' };
      }

      const result = await detectHighlightsWithTemplate({
        videoPath,
        audioSampleBase64,
        similarityThreshold: similarityThreshold != null ? Number(similarityThreshold) : undefined,
        minGapSeconds: minGapSeconds != null ? Number(minGapSeconds) : undefined,
        onProgress: sendProgress
      });

      return result;
    } catch (error) {
      logger.error('[IPC] detect-audio-highlights-with-template error:', error);
      return {
        success: false,
        error: error.message || 'Audio highlight detection failed.'
      };
    }
  });

  ipcMain.handle('save-highlight-template', async (_event, template) => {
    return saveTemplate(template);
  });

  ipcMain.handle('load-highlight-templates', async () => {
    return loadTemplates();
  });

  ipcMain.handle('load-highlight-template-by-name', async (_event, name) => {
    return loadTemplateByName(name);
  });

  ipcMain.handle('delete-highlight-template', async (_event, name) => {
    return deleteTemplate(name);
  });

  ipcMain.handle('export-highlight-template', async (_event, { name, outputPath }) => {
    return await exportTemplate(name, outputPath);
  });

  ipcMain.handle('import-highlight-template', async (_event, filePath) => {
    return importTemplate(filePath);
  });

  ipcMain.handle('extract-audio-sample', async (_event, payload) => {
    const { videoPath, templateStart, templateEnd } = payload || {};
    try {
      if (!videoPath || templateStart == null || templateEnd == null) {
        return { success: false, error: 'videoPath, templateStart, and templateEnd are required.', audioSample: null };
      }
      const { extractAudioToPcm } = require('./ffmpeg');
      const buffer = await extractAudioToPcm(videoPath, Number(templateStart), Number(templateEnd));
      if (!buffer || buffer.length < 1024) {
        return { success: false, error: 'Extraction failed or sample too short.', audioSample: null };
      }
      const audioSample = buffer.toString('base64');
      return { success: true, audioSample };
    } catch (error) {
      logger.error('[IPC] extract-audio-sample error:', error);
      return { success: false, error: error.message || 'Failed to extract audio sample.', audioSample: null };
    }
  });

  ipcMain.handle('extract-visual-region', async (_event, payload) => {
    const { videoPath, timestamp, region } = payload || {};
    try {
      if (!videoPath || timestamp == null || !region) {
        return { success: false, error: 'videoPath, timestamp, and region are required.' };
      }
      
      const result = await extractTemplateFromRegion(
        videoPath,
        Number(timestamp),
        region
      );
      
      return { success: true, template: result };
    } catch (error) {
      logger.error('[IPC] extract-visual-region error:', error);
      return { success: false, error: error.message || 'Failed to extract visual region.' };
    }
  });

  ipcMain.handle('detect-visual-highlights', async (event, payload) => {
    const { videoPath, templateBase64, region, method, threshold, frameSkip, batchSize, minGapSeconds } = payload || {};
    const webContents = event.sender;

    const sendProgress = (percent, message) => {
      try {
        if (webContents && !webContents.isDestroyed()) {
          webContents.send('visual-scan-progress', { percent, message });
        }
      } catch (err) {
        logger.warn('[IPC] Failed to send visual-scan-progress:', err && err.message);
      }
    };

    try {
      if (!videoPath || !templateBase64 || !region) {
        return { success: false, error: 'videoPath, templateBase64, and region are required.' };
      }

      const result = await detectVisualHighlights({
        videoPath,
        templateBase64,
        region,
        method: method || 'fast',
        threshold: threshold != null ? Number(threshold) : undefined,
        frameSkip: frameSkip != null ? Number(frameSkip) : undefined,
        batchSize: batchSize != null ? Number(batchSize) : undefined,
        minGapSeconds: minGapSeconds != null ? Number(minGapSeconds) : undefined,
        onProgress: sendProgress
      });

      return result;
    } catch (error) {
      logger.error('[IPC] detect-visual-highlights error:', error);
      return {
        success: false,
        error: error.message || 'Visual highlight detection failed.'
      };
    }
  });

  ipcMain.handle('detect-combined-highlights', async (event, payload) => {
    const { videoPath, audioTemplateBase64, visualTemplateBase64, visualRegion, audioThreshold, visualThreshold, method, frameSkip, batchSize, minGapSeconds } = payload || {};
    const webContents = event.sender;

    const sendProgress = (percent, message) => {
      try {
        if (webContents && !webContents.isDestroyed()) {
          webContents.send('combined-scan-progress', { percent, message });
        }
      } catch (err) {
        logger.warn('[IPC] Failed to send combined-scan-progress:', err && err.message);
      }
    };

    try {
      if (!videoPath) {
        return { success: false, error: 'videoPath is required.' };
      }

      const result = await detectCombinedHighlights({
        videoPath,
        audioTemplateBase64,
        visualTemplateBase64,
        visualRegion,
        audioThreshold: audioThreshold != null ? Number(audioThreshold) : undefined,
        visualThreshold: visualThreshold != null ? Number(visualThreshold) : undefined,
        method,
        frameSkip: frameSkip != null ? Number(frameSkip) : undefined,
        batchSize: batchSize != null ? Number(batchSize) : undefined,
        minGapSeconds: minGapSeconds != null ? Number(minGapSeconds) : undefined,
        onProgress: sendProgress
      });

      return result;
    } catch (error) {
      logger.error('[IPC] detect-combined-highlights error:', error);
      return {
        success: false,
        error: error.message || 'Combined highlight detection failed.'
      };
    }
  });

  async function getUniqueDestinationPath(targetDirectory, originalName) {
    const ext = path.extname(originalName);
    const baseName = path.basename(originalName, ext);
    let candidate = path.join(targetDirectory, originalName);
    let counter = 1;

    while (fs.existsSync(candidate)) {
      const suffix = ` (${counter})`;
      candidate = path.join(targetDirectory, `${baseName}${suffix}${ext}`);
      counter += 1;
    }

    return candidate;
  }
}

module.exports = {
  registerIpcHandlers
};

